(** Asn stands for Assertion. *)
Require Import Cmd.
Require Import Val.
Require Import Bio.
Require Import Op.
Require Import Heap.
Require Import Coq.Sets.Ensembles.
Require Import Coq.Arith.Peano_dec.

(** * Copredicate names *)
Inductive copred :=
| copred_ctr: nat -> copred
.

Theorem copred_eq_dec: forall p1 p2 : copred, {p1 = p2} + {p1 <> p2}.
Proof.
  destruct p1 as [n1].
  destruct p2 as [n2].
  destruct (eq_nat_dec n1 n2) ; auto.
  right. intro X. inversion X. contradiction.
Defined.

(** * Assertions *)
(* Note: asn_bio and asn_token expects a "val" instead of "place",
 * to make it usable together with asn_exists.
 *)
Inductive asn :=
| asn_val   : val -> asn
| asn_sepcon: asn -> asn -> asn
| asn_bio   : bio -> val -> val -> val -> val -> asn
| asn_no_op : val -> val -> asn
| asn_split : val -> val -> val -> asn
| asn_join  : val -> val -> val -> asn
| asn_token : val -> asn
| asn_copred: copred -> list val -> asn
| asn_disj  : asn -> asn -> asn
| asn_exists_: (val -> asn) -> asn
.

Notation "P &*& Q" := (asn_sepcon P Q) (right associativity, at level 19).
Definition asn_false := (asn_val (val_bool false)).
Notation emp := (asn_val (val_bool true)).
Notation "'asn_exists' x .. y , p" := (asn_exists_ (fun x => .. (asn_exists_ (fun y => p)) ..))
   (at level 200, x binder, y binder, right associativity).
Notation "'asn_if' v 'asn_then' a1 'asn_else' a2" :=
  (asn_disj
     (asn_sepcon (asn_val v) a1)
     (asn_sepcon (asn_val (unop_neg v)) a2)
  )
    (at level 19)
.

(** Example: there exists an x such that x < 7. *)
Definition asn_example1: asn :=
  (asn_exists x , asn_val (binop_lt x (val_int 7))).

(** Example: "token(?t1)" (without referring to t1 in postcondition) *)
Definition asn_example2: asn :=
  (asn_exists t1, asn_token t1).

Section models.
  Definition copred_def := (list val -> asn).
  Variable copred_defs : copred -> copred_def.

  (** The assertion is true for the given heap. *)
  CoInductive models : heap -> asn -> Prop :=
  | models_val:
      models emptyheap (asn_val (val_bool true))
  | models_bio: forall (b: bio) (t1 t2: val) (arg ret: val),
      models {| chunk_bio b t1 arg ret t2 |}
             (asn_bio b t1 arg ret t2)
  | models_join: forall (t1 t2 t3: val),
      models {| chunk_join t1 t2 t3 |} (asn_join t1 t2 t3)
  | models_split: forall (t1 t2 t3: val),
      models {| chunk_split t1 t2 t3 |} (asn_split t1 t2 t3)
  | models_token: forall (t1: val),
      models {| (chunk_token t1) |} (asn_token t1)
  | models_sepcon: forall (P Q: asn) (h: heap),
      (exists h1:heap, exists h2:heap, 
        heap_eq h (heap_plus h1 h2) /\ models h1 P /\ models h2 Q) ->
      models h (asn_sepcon P Q)
  | models_pred: forall (p: copred) (args: list val) (h: heap),
      models h ((copred_defs p) args) ->
      models h (asn_copred p args)
  | models_disj_left: forall (P Q: asn) (h: heap),
      models h P ->
      models h (asn_disj P Q)
  | models_disj_right: forall (P Q: asn) (h: heap),
      models h Q ->
      models h (asn_disj P Q)
  | models_exists: forall (body: val->asn) (h: heap),
      (exists v: val, models h (body v)) ->
      models h (asn_exists_ body)
  | models_no_op: forall (t1 t2: val),
      models ({| chunk_no_op t1 t2 |}) (asn_no_op t1 t2)
  .

  (** P "implies" P'
   (let's avoid the confusing "weaker" and "stronger" terminology)
   *)
  Definition impl (P P': asn) :=
    forall h:heap, models h P -> models h P'.

  Lemma impl_self : forall a: asn,
      impl a a.
  Proof.
    intros a h H.
    assumption.
  Qed.

End models.
