(** A colist is a list that can but does not have to be infinite. *)
CoInductive colist (A: Type) :=
| conil : colist A
| cocons : A -> colist A -> colist A
.

Global Arguments conil {_}.
Global Arguments cocons {_} _ _.
Notation "x ::: y" := (cocons x y)
                        (right associativity, at level 60).


Notation " <<< x ; .. ; y >>> " := (cocons x .. (cocons y conil) ..).


CoFixpoint colist_append {A: Type} (tau1 tau2: colist A) :=
  match tau1 with
  | conil => tau2
  | cocons h t => h ::: (colist_append t tau2)
  end
.
Notation "x +++ y" := (colist_append x y)
                        (right associativity, at level 60).

Inductive finite_colist {A: Type}: colist A -> Prop :=
| fin_empty : finite_colist (conil)
| fin_step  : forall (cl: colist A) (a: A), finite_colist cl ->
                                            finite_colist (cocons a cl)
.

Lemma colist_help {A: Type}: forall (cl: colist A),
    cl = match cl with 
         | conil => conil
         | cocons a b => cocons a b
         end.
Proof.
  destruct cl ; trivial.
Defined.



CoInductive colist_eq {A: Type} : colist A -> colist A -> Prop :=
| colist_eq_nil: colist_eq conil conil
| colist_eq_cons: forall (cl1 cl2: colist A) (a: A),
    colist_eq cl1 cl2 -> colist_eq (cocons a cl1) (cocons a cl2)
.

Theorem eq_impl_coeq: forall (A: Type) (cl: colist A), colist_eq cl cl.
Proof.
  cofix.
  intros A cl.
  rewrite (colist_help cl).
  rewrite (colist_help cl).
  destruct cl.
  - constructor.
  - constructor.  apply eq_impl_coeq. 
Qed.    

Theorem eq_impl_coeq': forall (A: Type) (cl1 cl2: colist A), cl1 = cl2 -> colist_eq cl1 cl2.
Proof.
  intros A cl1 cl2 eq.
  rewrite eq.
  apply eq_impl_coeq.
Qed.

Axiom coeq_impl_eq: forall (A: Type) (cl1 cl2: colist A), colist_eq cl1 cl2 -> cl1 = cl2.

Lemma append_conil: forall (A: Type) (cl: colist A), colist_eq (colist_append conil cl) cl.
Proof.
  cofix.
  intros A cl.
  rewrite (colist_help (colist_append conil cl)).
  rewrite (colist_help cl).
  simpl.
  destruct cl.
  - constructor.
  - constructor.
    apply eq_impl_coeq.
Qed.

Theorem append_conil_eq: forall {A: Type} {cl: colist A}, conil+++cl = cl.
Proof.
  intros A cl.
  apply coeq_impl_eq.
  apply append_conil.
Qed.

Lemma append_cocons: forall {A: Type} {cl: colist A} {a: A}, colist_eq (<<<a>>> +++ cl) (a:::cl).
Proof.
  intros.
  rewrite (colist_help (<<< a >>> +++ cl)).
  simpl.
  constructor.
  apply append_conil.
Qed.

Theorem append_cocons_eq: forall {A: Type} {cl: colist A} {a: A}, (<<<a>>> +++ cl) = (a:::cl).
Proof.
  intros.
  apply coeq_impl_eq.
  apply append_cocons.
Qed.

Theorem colist_append_assoc: forall (A: Type) (cl1 cl2 cl3: colist A),
    colist_eq ((cl1 +++ cl2) +++ cl3) (cl1 +++ (cl2 +++ cl3))
.
Proof.
  cofix.
  intros A cl1 cl2 cl.
  rewrite (colist_help (colist_append (colist_append cl1 cl2) cl)).
  rewrite (colist_help (colist_append cl1 (colist_append cl2 cl))).
  simpl.
  destruct cl1.
  - simpl.
    destruct cl2.
    + destruct cl.
      constructor.
      apply eq_impl_coeq.
    + constructor.
      apply eq_impl_coeq.
  - constructor.
    apply colist_append_assoc.
Qed.

Theorem colist_append_assoc_eq: forall (A: Type) (cl1 cl2 cl3: colist A),
    ((cl1 +++ cl2) +++ cl3) = (cl1 +++ (cl2 +++ cl3))
.
Proof.
  intros A cl1 cl2 cl3.
  apply coeq_impl_eq.
  apply colist_append_assoc.
Qed.


Theorem append_fin: forall (A: Type) (cl1 cl2: colist A),
    finite_colist cl1 -> finite_colist cl2 ->
    finite_colist (cl1 +++ cl2).
Proof.
  intros A cl1 cl2 fincl1 fincl2.
  induction fincl1.
  - rewrite colist_help. simpl.
    rewrite <- colist_help.
    assumption.
  - rewrite colist_help. simpl. constructor. assumption.
Qed.

Lemma diff_constructor: forall (A: Type) (a: A) (cl: colist A),
    conil = a:::cl -> False
.
Proof.
  intros A a cl1 eq.
  inversion eq.
Qed.

Lemma colist_help_app: forall (A: Type) (a: A) (cl1 cl2: colist A),
    colist_eq ((a:::cl1) +++ cl2) (a ::: (cl1 +++ cl2))
.
Proof.
  intros A a cl1 cl2.
  rewrite (colist_help ((a:::cl1)+++cl2)).
  simpl.
  constructor.
  apply eq_impl_coeq.
Qed.


Lemma colist_help_app_eq: forall (A: Type) (a: A) (cl1 cl2: colist A),
    ((a:::cl1) +++ cl2) = (a ::: (cl1 +++ cl2))
.
Proof.
  intros A a cl1 cl2.
  apply coeq_impl_eq.
  apply colist_help_app.
Qed.

Lemma append_conil_back: forall (A: Type) (cl: colist A), colist_eq (cl +++ conil) cl.
Proof.
  cofix.
  intros A cl.
  destruct cl.
  - apply append_conil.
  - rewrite colist_help_app_eq.
    constructor.
    apply append_conil_back.
Qed.

Lemma append_conil_back_eq: forall (A: Type) (cl: colist A), (cl +++ conil) = cl.
Proof.
  intros A cl.
  apply coeq_impl_eq.
  apply append_conil_back.
Qed.

Require Import Coq.Init.Datatypes.
Local Open Scope list_scope.

Inductive fin_witness {A: Type}: colist A -> list A -> Type :=
| finnil: fin_witness conil nil
| fincons: forall (a: A) (cl: colist A) (l: list A),
    fin_witness cl l -> fin_witness (a:::cl) (a::l)
.

Require Import Coq.Program.Equality.

Theorem append_fin': forall (A: Type) (cl1 cl2: colist A),
    finite_colist (cl1 +++ cl2) -> finite_colist cl1 /\ finite_colist cl2.
Proof.
  intros A cl1 cl2 fin_app.
  dependent induction fin_app. (* Alternative approach: search for "convoy pattern". *)
  - destruct cl1 as [ | hcl1 tcl1].
    + destruct cl2 as [ | hcl2 tcl2].
      * split ; constructor.
      * rewrite (append_conil_eq) in x.
        inversion x.
    + rewrite colist_help_app_eq in x.
      inversion x.
  - destruct cl1.
    + destruct cl2.
      * rewrite append_conil_eq in x.
        inversion x.
      * rewrite append_conil_eq in x.
        inversion x.
        split ; constructor.
        destruct H1.
        assumption.
    + rewrite colist_help_app_eq in x.
      inversion x.
      apply IHfin_app in H1.
      split ; try constructor; intuition.
Qed.


(** Note: never used, but you can define infiniteness of a
    colist like this: *)
CoInductive infinite_colist {A: Type}: colist A -> Prop :=
| infin     : forall (cl: colist A) (a: A), infinite_colist cl ->
                                            infinite_colist (cocons a cl)
.


Theorem app_notfin : forall (A: Type) (cl1 cl2: colist A),
    ~ finite_colist cl1 -> colist_eq (cl1 +++ cl2) cl1
.
Proof.
  cofix.
  intros A cl1 cl2 notfin.
  rewrite (colist_help (cl1+++cl2)).
  rewrite (colist_help cl1).
  simpl.
  destruct cl1.
  - destruct cl2.
    + constructor.
    + contradiction notfin.
      constructor.
  - constructor.
    apply app_notfin.
    intro H.
    contradiction notfin.
    constructor.
    assumption.
Qed.

Theorem app_notfin_eq: forall (A: Type) (cl1 cl2: colist A),
    ~ finite_colist cl1 -> (cl1 +++ cl2) = cl1
.
Proof.
  intros A cl1 cl2 notfin.
  apply coeq_impl_eq.
  apply app_notfin.
  assumption.
Qed.

CoFixpoint example (startwith: nat) : colist nat :=
  cocons startwith (example (startwith + 1))
.