(** Example of a program "infinite_counter"
    It prints 0,1,2,3,4,... on the screen and never stops.
    
    This example shows the big step relation and a heap for which the precondition holds.
*)

Require Import Cmd.
Require Import Asn.
Require Import Val.
Require Import Bio.
Require Import Op.
Require Import Hoare.
Require Import Trace.
Require Import Colist.
Require Import Coq.Arith.Peano_dec.
Require Import Heap.
Import Coq.Lists.List.ListNotations.
Local Open Scope list_scope.
Require Import ZArith.
Require Import Coq.Logic.FunctionalExtensionality.

Notation copred_inf_counter  := (copred_ctr 4).

(** Example predicate definitions. *)
Definition copred_defs (p: copred) : list val -> asn :=
  match p with
    | copred_inf_counter => 
      fun args =>
        match args with
        | [t1; i] => 
          asn_exists t2 r,
          asn_bio stdout t1 i r t2
          &*& asn_copred copred_inf_counter [t2; (binop_plus i (val_int 1))]
        | _ =>
          asn_false
      end
    | _ => fun _ => asn_false
  end
.


(* These are Notations instead of definitions such that we can put them in pattern matching. *)
Notation func_inf_counter := (func_ctr 4).

(** Function Context/definitions. *)
Definition fc (f: func) :=
  match f with
    | func_inf_counter =>
      fun (args: list val) =>
        match args with
        | [i] =>
          cmd_bio stdout i;;
          cmd_app func_inf_counter [binop_plus i (val_int 1)]
        | _ =>
          Cundefined
        end
    | _ => fun _ => Cundefined
  end
.


Definition inf_counter_precondition (t1: val) : asn :=
  asn_token t1 &*&
  asn_copred copred_inf_counter [t1; (val_int 0)]
.

(** Example program with it's contract. *)
Definition hoare_triple_quant_inf_counter : hoare_triple_quant :=
  fun (vs: list val) =>
    match vs with
      | [t1] => {{ inf_counter_precondition t1 }}
                         (fc func_inf_counter) [val_int 0]
                         {{ fun _ => asn_false }}
      | _ => {{ emp }} Cundefined {{ fun _ => emp }}
    end
.

(** * Big step relation example. *)

CoFixpoint expected_trace_inf_counter (i: Z) :=
  action_ctr stdout (val_int i) (val_int 0) :::
             action_no_io :::
             expected_trace_inf_counter (i+1)
.


(** Example of an execution of the program "inf_counter", i.e. of the
    step relation. Note that this is (and should be) an infinite execution. *)
Theorem inf_counter_step:
  forall (i: Z), big_step fc
           ((fc func_inf_counter) [val_int i])
           (expected_trace_inf_counter i)
           (val_int 0).
Proof.
  cofix.
  intro i.
  rewrite (colist_help (expected_trace_inf_counter i)).
  simpl.
  rewrite <- append_cocons_eq.
  eapply step_let.
  - constructor.
  - constructor.
    apply inf_counter_step.
Qed.

(** * Example of model relation *)

Lemma sumbool_and_dec: forall {A B: Prop}, {A}+{~A} -> {B}+{~B} -> ({A/\B}+{~(A/\B)}).
Proof.
  intros.
  destruct H; destruct H0; try (right; intro contr;  destruct contr; contradiction; fail).
  left.
  split; assumption.
Qed.

Notation "A /\\ B" := (sumbool_and_dec A B) (right associativity, at level 12).

Definition chunk_wellformed (c: chunk) : Prop :=
  match c with
  | chunk_bio b (val_place (place_ctr t1_n))
              (val_int i)
              (val_int 0)
              (val_place (place_ctr (t2_n))) =>
    b = stdout /\ t2_n = (t1_n + 1)%Z /\ i = t1_n
  | _ => False
  end
.

Definition chunk_get_t1 (c: chunk) : Z :=
  match c with
  | chunk_bio b (val_place (place_ctr t1_n)) _ _ _ => t1_n
  | _ => 0%Z
  end
.


Definition good_chunk (startwith: Z) (c: chunk) : Prop :=
  chunk_wellformed c /\ (startwith <= chunk_get_t1 c)%Z
.

Lemma good_chunk_dec: forall (startwith: Z) (c: chunk),
    {good_chunk startwith c} + {~ good_chunk startwith c}.
Proof.
  unfold good_chunk.
  unfold chunk_wellformed.
  intros.
  destruct c ; try (right; intro H; destruct H; contradiction H; fail).
  destruct v ; try (right; intro H; destruct H; contradiction H; fail).
  destruct v0; try (try destruct p; try destruct b0; try destruct l; try destruct f;
                    right; intro H;destruct H;  contradiction H; fail).
  destruct v1; try (try destruct p; try destruct b0; try destruct l; try destruct f;
                    right; intro H; destruct H; contradiction H; fail).
  destruct v2; try (try destruct p; try destruct b0; try destruct l; try destruct f; try destruct z0;
                    right; intro H; destruct H;  contradiction H; fail).
  destruct p.
  destruct z0 ; try (right; intro X; destruct X as [X X']; apply X; fail).
  destruct p0.
  simpl.
  destruct (((bio_eq_dec b stdout)
         /\\ (Z.eq_dec z0 (z1 + 1))
         /\\ (Z.eq_dec z z1))
         /\\ (Z_le_dec startwith z1)).
  - left.
    assumption.
  - right.
    assumption.
Qed.

Definition heap_inf_counter_helper (startwith: Z) (c: chunk) : natinf :=
  if good_chunk_dec startwith c then
    fin 1
  else
    fin 0
.

Definition heap_inf_counter: heap :=
  heap_plus {| chunk_token (val_place (place_ctr 0)) |} (heap_inf_counter_helper 0)
.

Axiom excluded_middle : forall P: Prop, P + ~P.

Lemma de_morgan_not_and : forall (A B: Prop), ~(A /\ B) -> ~A \/ ~B.
Proof.
  intros.
  destruct (excluded_middle A).
  - destruct (excluded_middle B).
    + contradiction H.
      split; assumption.
    + right.
      assumption.
  - left.
    assumption.
Qed.

Lemma heap_inf_counter_split_eq: forall (n: Z),
    forall (c: chunk),
      c = chunk_bio stdout (val_place (place_ctr n)) (val_int n) (val_int 0)
                      (val_place (place_ctr (n + 1))) ->
      heap_eq (heap_inf_counter_helper n) (heap_plus {|c|} (heap_inf_counter_helper (n+1))).
Proof.
  intros.
  apply heap_eq_iff_eq.
  apply functional_extensionality.
  intro x.
  unfold heap_inf_counter_helper.
  unfold heap_add.
  unfold heap_plus.
  unfold heap_singleton.
  unfold emptyheap.
  destruct (good_chunk_dec (n) x).
  - destruct (good_chunk_dec (n+1) x).
    + destruct (chunk_eq_dec c x).
      * unfold good_chunk in g0.
        destruct g0.
        rewrite <- e in H1.
        rewrite H in H1.
        simpl in H1.
        assert False.
        { omega. }
        contradiction.
      * reflexivity.
    + destruct (chunk_eq_dec c x).
      * reflexivity.
      * unfold good_chunk in g.
        unfold good_chunk in n0.
        destruct g.
        apply de_morgan_not_and in n0.
        destruct n0.
        { contradiction. }
        { 
          assert (n = chunk_get_t1 x) as H_eq. { omega. }
          unfold chunk_wellformed in H0.
          destruct x; try destruct v; try destruct p; try destruct v0;
          try destruct v1; try destruct z1; try destruct v2;
          try destruct p; try contradiction.
          unfold chunk_get_t1 in H2.
          destruct H0.
          destruct H3.
          subst.
          contradiction n1.
          reflexivity.
        } 
  - destruct (good_chunk_dec (n+1) x).
    + unfold good_chunk in n0.
      unfold good_chunk in g.
      destruct g.
      apply de_morgan_not_and in n0.
      destruct n0.
      { contradiction. }
      { omega. }
    + destruct (chunk_eq_dec c x).
      * rewrite <- e in n0.
        rewrite H in n0.
        contradiction n0.
        unfold good_chunk.
        split.
        unfold chunk_wellformed.
        split; try split; try reflexivity.
        reflexivity.
      * reflexivity.
Qed.

Lemma pred_models: forall (n: Z),
   @models copred_defs (heap_inf_counter_helper n)
     (copred_defs copred_inf_counter [val_place (place_ctr n); val_int n]).
Proof.
  cofix.
  constructor.
  exists (val_place (place_ctr (n+1))).
  constructor.
  exists (val_int 0).
  constructor.
  exists {| chunk_bio stdout (val_place (place_ctr n)) (val_int n) (val_int 0) (val_place (place_ctr (n+1)))|}.
  exists (heap_inf_counter_helper (n+1)).
  split; try split.
  - eapply heap_inf_counter_split_eq.
    reflexivity.
  - constructor.
  - assert ( binop_plus (val_int n) (val_int 1) = val_int (n+1)) as H_eq.
    { unfold binop_plus.
      reflexivity.
    }
    rewrite H_eq.
    constructor.
    apply pred_models with (n := (n+1)%Z).
Qed.

(** Example: the assertion (of the precondition) holds for the heap.
    Note that the heap is infinite. *)
Theorem inf_counter_precond_models: 
    @models copred_defs heap_inf_counter (inf_counter_precondition (val_place (place_ctr 0))).
Proof.
  constructor.
  exists {|chunk_token (val_place (place_ctr 0))|}.
  exists (heap_inf_counter_helper 0).
  split ; try split.
  - constructor.
  - apply models_pred.
    apply pred_models.
Qed.


(** * Negative example of model relation *)

Definition t1: val := val_place (place_ctr 0).
Definition t2: val := val_place (place_ctr 1).
Definition chunk_stdout := chunk_bio stdout t1 (val_int 0) (val_int 0) t2.
Definition bad_heap := {| chunk_token t1,
                          chunk_stdout |}.

Definition the_asn := asn_token t1 &*& asn_bio stderr t1 (val_int 0) (val_int 0) t2.

Theorem bad_heap_not_a_model_for_the_asn: ~ models copred_defs bad_heap the_asn.
Proof.
  intro H.
  inversion H ; clear H ; subst.
  destruct H2 as [h_token [h_bio [h_eq [models_token models_bio]]]].
  inversion models_token.
  inversion models_bio.
  unfold heap_eq in h_eq.
  subst.
  assert (bad_heap chunk_stdout =
          (heap_plus {|chunk_token t1|}
                     {|chunk_bio stderr t1 (val_int 0) (val_int 0) t2|}) chunk_stdout) as H3. {
    apply h_eq.
  }
  inversion H3.
Qed.


Theorem inf_counter_precond_not_models:
  ~ @models copred_defs emptyheap (inf_counter_precondition t1)
.
Proof.
  intro H.
  inversion H; subst.
  destruct H2 as [h1 [h2 [h_eq [mod_h1 mod_h2]]]].
  inversion mod_h1.
  subst.
  
  assert (emptyheap (chunk_token t1) = (heap_plus {|chunk_token t1|} h2) (chunk_token t1)) as H2.
  { apply h_eq. }
  unfold emptyheap in H2.
  unfold heap_add in H2.
  unfold heap_singleton in H2.
  unfold heap_plus in H2.
  simpl in H2.
  unfold natinf_plus in H2.
  simpl in H2.
  destruct (h2 (chunk_token t1)); inversion H2.
Qed.

