(** Example of a program "tee". "tee" reads input, and
    writes what it reads to stdout and stderr, but its contract
    allow buffering.

    This example verifies one function ("tee_out") of the tee program
    and shows the big step relation holds for that function,
    and a certain trace.
*)

Require Import Cmd.
Require Import Asn.
Require Import Val.
Require Import Bio.
Require Import Op.
Require Import Hoare.
Require Import Trace.
Require Import Colist.
Require Import Coq.Arith.Peano_dec.
Import Coq.Lists.List.ListNotations.
Local Open Scope list_scope.
Require Import ImplMoveAsn.
Require Import Impl.
Require Import Heap.

(* These are Notations instead of definitions such that we can
   use them in pattern matching. *)
Notation copred_tee_out  := (copred_ctr 0).
Notation copred_tee_outs := (copred_ctr 1).
Notation copred_tee      := (copred_ctr 2).
Notation copred_reads    := (copred_ctr 3).

(** Copredicate definitions used in the contract of the example program. *)
Definition copred_defs (p: copred) : list val -> asn :=
  match p with
    | copred_tee_out => 
      fun args =>
        match args with
        | [t1; c; t2] => 
          asn_exists tp1 tp2 tr1 tr2 _1 _2,
          (asn_split t1 tp1 tr1
          &*& asn_bio stdout tp1 c _1 tp2
          &*& asn_bio stderr tr1 c _2 tr2
          &*& asn_join tp2 tr2 t2)
        | _ =>
          asn_false
      end
    | copred_tee_outs =>
      fun args =>
        match args with
        | [t1; text; t2] =>
          asn_exists t_out,
          ((asn_if (binop_eq text (val_list_int [])) asn_then
            (asn_val (binop_eq t1 t2))
          asn_else
            (asn_copred copred_tee_out [t1; unop_head text; t_out]
            &*& asn_copred copred_tee_outs [t1; unop_tail text; t2])
          ))
        | _ =>
          asn_false
        end
    | copred_tee =>
      fun args =>
        match args with
        | [t1; text; t2] =>
          asn_exists tr1 tw1 tr2 tw2,
          asn_split t1 tr1 tw1
          &*& asn_copred copred_reads [tr1; text; tr2]
          &*& asn_copred copred_tee_outs [tw1; text; tw2]
          &*& asn_join tr2 tw2 t2
        | _ =>
          asn_false
        end
    | copred_reads =>
      fun args =>
        match args with
        | [t1; text; t3] =>
          asn_exists r t2 sub,
          asn_bio stdin t1 (val_int 0) r t2
          &*& asn_if (binop_lt r (val_int 0)) asn_then
            asn_val (binop_eq text (val_list_int nil))
          asn_else
            asn_copred copred_reads [t2; sub; t3]
            &*& asn_val (binop_eq text (binop_cons r sub))
        | _ =>
          asn_false
        end
    | _ => fun _ => asn_false
  end
.

Notation func_tee_out := (func_ctr 0).
Notation func_main := (func_ctr 1).
Notation func_tee := (func_ctr 2).

(** Function Context/definitions. *)
Definition fc (f: func) :=
  match f with
    | func_tee_out =>
      fun (args: list val) =>
        match args with
        | [c] => (cmd_bio stdout c) ;;
                 (cmd_bio stderr c)
        | _ => Cundefined
        end
    | func_tee =>
      fun (args: list val) =>
        Clet c1 := (cmd_bio stdin (val_int 0)) in 
        cmd_if (binop_gte c1 (val_int 0)) (
          Clet c2 := (cmd_bio stdin (val_int 0)) in
          cmd_app func_tee_out [c1] ;;
          cmd_if (binop_gte c2 (val_int 0)) (
            cmd_app func_tee_out [c2] ;;
            cmd_app func_tee []
          )(
            cmd_skip
          )
        )(
          cmd_skip
        )
    | func_main =>
      fun (args: list val) =>
        cmd_app func_tee []
    | _ => fun _ => Cundefined
  end
.

(** Example program "tee_out" with it's contract. *)
Definition hoare_triple_quant_tee_out: hoare_triple_quant :=
  fun (vs: list val) =>
    match vs with
      | [t1; c; t2] => {{ asn_token t1 &*&
                                    asn_copred copred_tee_out [t1; c; t2] }}
                         (fc func_tee_out) [c]
                         {{ fun _ => asn_token t2 }}
      | _ => {{ emp }} Cundefined {{ fun _ => emp }}
    end
.

(** Example: verifying a program ("tee_out") using the proof rules. *)
Theorem proof_tee_out : forall (vs: list val), @proof copred_defs fc (hoare_triple_quant_tee_out vs).
Proof.
  intro vs.
  do 4 (try destruct vs ; try apply proof_emp_undefined).
  rename v into t1; rename v0 into c; rename v1 into t2.
  apply (apply_impl (impl_replace_back (models_open copred_tee_out))).
  apply (apply_impl impl_sepcon_comm).
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro tp1.
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro tp2.
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro tr1.
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro tr2.
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro _1.
  apply (apply_impl impl_exists_longer); apply proof_exists_pre; intro _2.
  apply (apply_impl impl_sepcon_comm).
  apply do_split.

  apply (apply_impl impl_sepcon_comm_2nd); simpl.
  apply (apply_impl (impl_replace_back impl_sepcon_assoc2)).
  apply (apply_impl (impl_replace_back (impl_replace_back impl_sepcon_assoc2))).
  apply (apply_impl impl_sepcon_assoc).
  apply proof_let with (Q1 := fun _ => asn_token tp2 &*&
                                       asn_bio stderr tr1 c _2 tr2 &*&
                                       asn_join tp2 tr2 t2 &*&
                                       asn_token tr1).
  * apply proof_frame.

    eapply proof_rewrite_post.
    {
      intro r.
      apply asn_rewrite_impl.
      apply (impl_add_eq r _1).
    }
    apply proof_bio. 
  * intro _3.
    eapply proof_rewrite_post.
    { intro _4.
      apply asn_rewrite_impl.
      apply impl_add_emp_back.
    }
    simpl.
    eapply proof_rewrite_pre.
    { apply asn_rewrite_impl.
      apply impl_self.
    }
    eapply proof_rewrite_post. {
      intro v.
      apply asn_rewrite_join.
    }
    eapply proof_rewrite_post. {
      intro v.
      apply asn_rewrite_impl.
      intros h H.
      remove_emp_back_lt. simpl.
      apply impl_sepcon_comm_2nd.
      apply H.
    }
    apply proof_rewrite_post with (Q := fun _ => (asn_join tp2 tr2 t2 &*& asn_token tr2) &*& asn_token tp2).
    { intro _4.
      apply asn_rewrite_impl.
      apply impl_sepcon_comm.
    }
    eapply proof_rewrite_post.
    { intro _4.
      apply asn_rewrite_impl.
      apply (impl_replace_front impl_sepcon_comm).
    }
    eapply proof_rewrite_post.
    { intro _4.
      apply asn_rewrite_impl.
      apply impl_sepcon_assoc.
    }

    do 2 apply (apply_impl impl_sepcon_assoc).
    apply (apply_impl impl_sepcon_comm).
    apply (apply_impl (impl_replace_back (impl_replace_front impl_sepcon_comm))).
    apply (apply_impl (impl_replace_back impl_sepcon_assoc2)).
    apply (apply_impl (impl_replace_back (impl_replace_back impl_sepcon_comm))).
    apply (apply_impl impl_sepcon_assoc).
    
    apply proof_frame.


    eapply proof_rewrite_post.
    {
      intro r.
      apply asn_rewrite_impl.
      apply (impl_add_eq r _2).
    }

    apply proof_bio.
Qed.


(** Example of an execution of the program "tee_out", i.e. of the
    step relation. *)
Theorem tee_out_step:
  big_step fc
           ((fc func_tee_out) [val_int 42])
           (<<< action_ctr stdout (val_int 42) (val_int 0) ;
            action_ctr stderr (val_int 42) (val_int 0) >>>)
           (val_int 0).
Proof.
  simpl.
  assert (<<< action_ctr stdout (val_int 42) (val_int 0) >>> +++
          (action_ctr stderr (val_int 42) (val_int 0):::conil)
          = 
          <<< action_ctr stdout (val_int 42) (val_int 0);
          action_ctr stderr (val_int 42) (val_int 0) >>>) as H.
  {
    apply coeq_impl_eq.
    rewrite (colist_help (<<< action_ctr stdout (val_int 42) (val_int 0) >>> +++
      <<< action_ctr stderr (val_int 42) (val_int 0) >>>)).
    simpl.
    constructor.
    apply append_conil.
  }
  rewrite <- H.
  apply step_let with (v1 := val_int 0).
  { constructor. }
  constructor.
Qed.



(** Example program "tee" with it's contract. *)
Definition hoare_triple_quant_tee: hoare_triple_quant :=
  fun (vs: list val) =>
    match vs with
      | [t1; text; t2] => {{ asn_token t1 &*&
                          asn_copred copred_tee [t1; text; t2] }}
                         (fc func_tee) [text]
                         {{ fun _ => asn_token t2 }}
      | _ => {{ emp }} Cundefined {{ fun _ => emp }}
    end
.
