Require Import Val.
Require Import Bio.
Require Import Coq.Logic.FunctionalExtensionality.
Require Import Coq.Arith.Plus.

Inductive natinf :=
| fin: nat -> natinf
| inf: natinf
.

Definition natinf_plus (ni1 ni2: natinf) :=
  match (ni1, ni2) with
  | (fin n1, fin n2) => fin (n1 + n2)
  | _ => inf
  end
.

Theorem natinf_plus_commutative: forall (ni1 ni2: natinf),
    natinf_plus ni1 ni2 = natinf_plus ni2 ni1.
Proof.
  intros ni1 ni2.
  destruct ni1 ; auto.
  - destruct ni2 ; auto.
    + assert (natinf_plus (fin n) (fin n0) = fin (n + n0)) as H1 ; auto.
      rewrite H1.
      rewrite plus_comm.
      auto.
  - destruct ni2 ; auto.
Qed.

Theorem natinf_plus_assoc: forall (ni1 ni2 ni3: natinf),
    natinf_plus (natinf_plus ni1 ni2) ni3 = natinf_plus ni1 (natinf_plus ni2 ni3).
Proof.
  intros ni1 ni2 ni3.
  destruct ni1 ; destruct ni2 ; destruct ni3 ; try reflexivity.
  unfold natinf_plus.
  rewrite plus_assoc.
  reflexivity.
Qed.

Inductive chunk :=
| chunk_token: val -> chunk
| chunk_join: val -> val -> val -> chunk
| chunk_split: val -> val -> val -> chunk
| chunk_bio: bio -> val -> val -> val -> val -> chunk
| chunk_no_op: val -> val -> chunk
.

Ltac SolveEq :=
  solve [right; intro X; inversion X; contradiction
        |left; subst; reflexivity]
.

Theorem chunk_eq_dec : forall c1 c2 : chunk, {c1 = c2} + {c1 <> c2}.
Proof.
  intros c1 c2.

  induction c1; induction c2;
  (* join and split *)
  try (destruct (val_eq_dec v v2);
       destruct (val_eq_dec v0 v3);
       destruct (val_eq_dec v1 v4); SolveEq);
  (* different constructor. *)
  try SolveEq.
  - (* token *)
    destruct (val_eq_dec v v0) ; SolveEq.
  - (* bio *)
    destruct (bio_eq_dec b b0);
    destruct (val_eq_dec v v3);
    destruct (val_eq_dec v0 v4);
    destruct (val_eq_dec v1 v5);
    destruct (val_eq_dec v2 v6); SolveEq.
  - (* no_op *)
    destruct (val_eq_dec v v1);
    destruct (val_eq_dec v0 v2); SolveEq.
Defined.

Definition heap := (chunk -> natinf).

Definition heap_eq (h1 h2: heap) := forall (c: chunk), h1 c = h2 c.

Lemma heap_eq_iff_eq: forall (h1 h2: heap), heap_eq h1 h2 <-> h1 = h2.
Proof.
  intros h1 h2.
  split.
  - apply functional_extensionality.
  - intros eq.
    intros x.
    rewrite eq.
    trivial.
Qed.

Definition emptyheap: heap := fun (_:chunk) => fin 0.

Definition heap_plus (h1: heap) (h2: heap) : heap :=
  fun chunk_request => natinf_plus (h1 chunk_request) (h2 chunk_request)
.


Theorem heap_plus_commutative : forall (h1 h2: heap), heap_plus h1 h2 = heap_plus h2 h1.
Proof.
  intros h1 h2.
  apply functional_extensionality.
  intro x.
  unfold heap_plus.
  rewrite natinf_plus_commutative.
  reflexivity.
Qed.

Theorem heap_plus_assoc : forall (h1 h2 h3: heap),
    heap_plus (heap_plus h1 h2) h3 = heap_plus h1 (heap_plus h2 h3).
Proof.
  intros h1 h2 h3.
  apply functional_extensionality.
  intro x.
  unfold heap_plus.
  apply natinf_plus_assoc.
Qed.


Theorem heap_plus_empty: forall (h: heap), heap_plus emptyheap h = h.
Proof.
  intros h.
  apply functional_extensionality.
  intro x.
  unfold heap_plus.
  destruct (h x) ; auto.
Qed.

Theorem heap_plus_empty_right: forall (h: heap), heap_plus h emptyheap = h.
Proof.
  intro h.
  rewrite heap_plus_commutative.
  apply heap_plus_empty.
Qed.


Definition heap_singleton (c: chunk) : heap :=
  fun (chunk_request : chunk) =>
    if chunk_eq_dec c chunk_request then
      fin 1
    else
      fin 0
. 

Definition heap_add (c: chunk) (h: heap) : heap :=
  heap_plus (heap_singleton c) h.

Notation "{| x , .. , y |}" := (heap_add x .. (heap_add y emptyheap) ..) (at level 0).

Theorem heap_add_empty: forall (c: chunk), heap_add c emptyheap = heap_singleton c.
Proof.
  intro c.
  unfold heap_add.
  rewrite <- heap_plus_empty.
  rewrite heap_plus_commutative.
  reflexivity.
Qed.

(** Example usage: heap_plus {| c1 |} {| c2 , c3 |} = {| c1, c2, c3 |} *)
Theorem heap_plus_one:
  forall (c: chunk) (h: heap),
  heap_plus ({| c |}) h = heap_add c h
.
Proof.
  intros c h.
  rewrite heap_add_empty.
  apply functional_extensionality.
  intro x.
  destruct (heap_singleton c x).
  - destruct (h x).
    + reflexivity.
    + reflexivity.
  - reflexivity.
Qed.
