Require Import Asn.
Require Import Val.
Require Import Cmd.
Require Import Heap.
Require Import Trace.
Require Import Colist.
Require Import Bio.
Require Import Op.

(** Hoare triple without quantification. *)
Inductive hoare_triple: Set :=
| hoare_triple_ctr: asn -> cmd -> (val -> asn) -> hoare_triple
.

Notation "{{ P }} c {{ Q }}" := (hoare_triple_ctr P c Q) (at level 20).

(** Hoare triple with quantification. *)
Definition hoare_triple_quant := (list val -> hoare_triple).

Section hoare.
  Context {copred_defs: copred -> copred_def}.
  Context {fc: (func -> (list val -> cmd))}.

  Definition valid_helper (h: heap) (a: asn)
             (tau: trace)
    :=
      ex (fun (g: heap + bottom) =>
            (traces h tau g) /\
            match g with
            | inl h' => (finite_colist tau -> models copred_defs h' a)
            | _ => True
            end
         )
  .

  (** When is a Hoare triple true. *)
  Definition valid (PcQ: hoare_triple) :=
    match (PcQ) with
      {{P}}c{{Q}} =>
      forall (h: heap),
        models copred_defs h P ->
        forall (tau: trace) (v: val),
          big_step fc c tau v ->
          valid_helper h (Q v) tau
    end
  .

  Inductive asn_rewrite : asn -> asn -> Set :=
  | asn_rewrite_impl: forall (a1 a2: asn), impl copred_defs a1 a2 -> asn_rewrite a1 a2
  | asn_rewrite_split : forall (t1 t2 t3: val) (P: asn),
      asn_rewrite (asn_token t1 &*& asn_split t1 t2 t3 &*& P) (asn_token t2 &*& asn_token t3 &*& P)
  | asn_rewrite_join : forall (t1 t2 t3: val) (P: asn),
      asn_rewrite (asn_token t1 &*& asn_token t2 &*& asn_join t1 t2 t3 &*& P) (asn_token t3 &*& P)
  .
  
  (** Proof rules *)
  Inductive proof : hoare_triple -> Set :=
  (* To prove e.g. { emp } 7 { fun x => x > 6 } *)
  | proof_val: forall (v: val) (f: val -> val),
      (f v) = val_bool true ->
      proof ({{ emp }}
               (cmd_val v)
               {{ (fun x => asn_val (f x)) }})
  | proof_let: forall (P: asn) (Q1 Q: val->asn) (c: cmd) (f:val->cmd),
      proof ({{P}} c {{Q1}}) ->
      (forall v: val, proof ({{Q1 v}} f v {{Q}})) ->
      proof ({{P}} cmd_let c f {{Q}})
  (* Note: this does not support non-terminating recursion. *)
  | proof_app: forall (P: asn) (Q: val->asn) (f: func) (vs: list val),
      proof ({{P}} (fc f) vs {{Q}}) ->
      proof ({{P}} cmd_app f vs {{Q}})
  | proof_then: forall (P: asn) (Q: val->asn) (ct ce: cmd),
      proof ({{P}} ct {{Q}})->
      proof ({{P}} cmd_if (val_bool true) ct ce {{Q}})
  | proof_else: forall (P: asn) (Q: val->asn) (v: val) (ct ce: cmd),
      v <> val_bool true ->
      proof ({{P}} ce {{Q}})->
      proof ({{P}} cmd_if v ct ce {{Q}})
  | proof_frame: forall (P: asn) (Q: val->asn) (R: asn) (c: cmd),
      proof ({{P}} c {{Q}}) ->
      proof ({{P &*& R}} c {{fun v => Q v &*& R}})
  | proof_leak: forall (P: asn) (Q: val->asn) (R: asn) (c: cmd),
      proof ({{P}} c {{fun v => Q v &*& R}}) ->
      proof ({{P}} c {{Q}})
  | proof_bio: forall (b: bio) (t1 t2: val) (v vr: val),
      proof ( {{asn_token t1 &*& asn_bio b t1 v vr t2}}
                cmd_bio b v
                {{fun x => (asn_val (binop_eq x vr)) &*& asn_token t2}} )
  | proof_noop: forall (P: asn) (Q: val->asn) (c: cmd) (t1 t2: val),
      proof ({{asn_token t2 &*& P}} c {{Q}}) ->
      proof ({{asn_token t1 &*& asn_no_op t1 t2 &*& P}} c {{Q}})
  | proof_rewrite: forall (P P': asn) (Q Q': val->asn) (c: cmd),
      asn_rewrite P P' ->
      (forall v: val, asn_rewrite (Q v) (Q' v)) ->
      proof ({{P'}} c {{Q}}) ->
      proof ({{P}} c {{Q'}})
  | proof_exists_pre: forall (vx: val -> asn) (Q: val->asn) (c:cmd),
      (forall (v: val), proof ({{vx v}} c {{Q}})) ->
      proof ({{asn_exists_ vx}} c {{Q}})
  | proof_disjunction: forall (P1 P2: asn) (Q: val->asn) (c: cmd),
    proof ({{P1}} c {{Q}}) ->
    proof ({{P2}} c {{Q}}) ->
    proof ({{asn_disj P1 P2}} c {{Q}})

  | proof_skip :
    proof ({{emp}} cmd_skip {{fun _ => emp}})
  | proof_false: 
      forall (c: cmd) (Q: val->asn),
      proof ({{asn_val (val_bool false)}}c{{Q}})
  .
  

  (** Helper that makes the use of proof_impl easier. *)
  Lemma apply_impl: forall {P P': asn} {Q: val->asn} {c: cmd}
      (use_impl: impl copred_defs P P'), proof ({{P'}}c{{Q}}) -> proof ({{P}}c{{Q}}).
  Proof.
    intros P P' Q c Hi Hp.
    apply proof_rewrite with (P':=P') (Q:=Q).
    - constructor.
      apply Hi.
    - intro v.
      constructor.
      apply impl_self.
    - assumption.
  Qed.

  Lemma proof_rewrite_post : forall {P: asn} {c: cmd} {Q': val->asn} (Q: val->asn),
      (forall v: val, asn_rewrite (Q v) (Q' v)) ->
      proof ({{P}}c{{Q}}) ->
      proof ({{P}}c{{Q'}}).
  Proof.
    intros P c Q' Q Ha Hp.
    apply proof_rewrite with (P' := P) (Q := Q) ; try assumption.
    apply asn_rewrite_impl.
    apply impl_self.
  Qed.

  Lemma proof_rewrite_pre : forall {P: asn} {c: cmd} {Q: val->asn} (P': asn),
      asn_rewrite P P' ->
      proof ({{P'}}c{{Q}}) ->
      proof ({{P}}c{{Q}}).
  Proof.
    intros P c Q P' Ha Hp.
    apply proof_rewrite with (P' := P') (Q := Q) ; try assumption.
    intro v.
    apply asn_rewrite_impl.
    apply impl_self.
  Qed.


  Lemma do_split: forall {t1 t2 t3: val} {P: asn} {c:cmd} {Q: val->asn},
      proof ({{asn_token t2 &*& asn_token t3 &*& P}}c{{Q}}) ->
      proof ({{asn_token t1 &*& asn_split t1 t2 t3 &*& P}}c{{Q}}).
  Proof.
    intros t1 t2 t3 P c Q H.

    eapply proof_rewrite.
    { apply asn_rewrite_split. }
    { intro v.
      apply asn_rewrite_impl.
      apply impl_self.
    }
    assumption.
  Qed.


  Lemma proof_emp_undefined : proof ({{emp}}Cundefined {{fun _ : val => emp}}).
  Proof.
    apply proof_val.
    reflexivity.
  Qed.
End hoare.


