(** Some assertion implications useful for examples. *)
Require Import Asn.
Require Import Val.
Require Import Heap.
Require Import Op.

Section impl_proofs.
  Context {copred_defs: copred -> copred_def}.

  Lemma models_open : forall (p: copred) {vs: list val},
      impl copred_defs (asn_copred p vs) ( (copred_defs p) vs ).
  Proof.
    intros p vs h H.
    inversion H.
    assumption.
  Qed.

  Lemma models_close : forall (p: copred) (vs: list val),
      impl copred_defs ((copred_defs p) vs) (asn_copred p vs).
  Proof.
    intros p vs h H.
    constructor.
    assumption.
  Qed.

  Lemma impl_id :
    forall {a: asn},
    impl copred_defs a a.
  Proof.
    intros a h H.
    apply H.
  Qed.

  Lemma impl_sepcon : forall {a a' r: asn},
      @impl copred_defs a a' ->
      @impl copred_defs (r &*& a) (r &*& a').
  Proof.
    intros r a a' H1.
    constructor.
    inversion H.
    destruct H3 as [h1 [h2 H3]].
    exists h1. exists h2.
    split; try split ; try apply H3.
    apply H1.
    apply H3.
  Qed.

  Lemma impl_sepcon_comm : forall {a1 a2: asn},
      impl copred_defs (a1 &*& a2) (a2 &*& a1).
  Proof.
    unfold impl.
    intros.
    inversion H.
    destruct H2 as [h1 [h2 [h_eq [mod_h1 mod_h2]]]].
    constructor.
    exists h2.
    exists h1.
    split; try split ; try assumption.
    rewrite heap_plus_commutative.
    assumption.
  Qed.

  Lemma impl_sepcon_assoc : forall {a1 a2 a3: asn},
      impl copred_defs (a1 &*& (a2 &*& a3)) ((a1 &*& a2) &*& a3).
  Proof.
    intros a1 a2 a3 h Hh.
    inversion Hh.
    destruct H1 as [h1 [h23 [Heq [Hh1 Hh23]]]].
    subst.
    inversion Hh23.
    destruct H1 as [h2 [h3 [Heq' [Hh2 Hh3]]]].
    subst.
    constructor.
    exists (heap_plus h1 h2).
    exists h3.
    split ; try split ; try assumption.
    - (* h = (h1 + h2) + h3 *)
      rewrite heap_plus_assoc.
      rewrite heap_eq_iff_eq in Heq'.
      rewrite <- Heq'.
      assumption.
    - (* h1+h2 |= a1 &*& a2 *)
      constructor.
      exists h1.
      exists h2.
      split ; try split ; auto.
  Qed.

  Lemma impl_sepcon_assoc2 :
    forall {a1 a2 a3: asn},
      impl copred_defs ((a1 &*& a2) &*& a3) (a1 &*& (a2 &*& a3)).
  Proof.
    intros a1 a2 a3 h Hh.
    inversion Hh.
    destruct H1 as [h12 [h3 [Heq [Hh12 Hh3]]]].
    subst.
    inversion Hh12.
    destruct H1 as [h1 [h2 [Heq' [Hh1 Hh2]]]].
    subst.
    constructor.
    exists h1.
    exists (heap_plus h2 h3).
    split ; try split ; try assumption.
    - (* h = h1 + (h2 + h3) *)
      rewrite <- heap_plus_assoc.
      rewrite heap_eq_iff_eq in Heq'.
      rewrite <- Heq'.
      assumption.
    - (* h2+h3 |= a2 &*& a3 *)
      constructor.
      exists h2.
      exists h3.
      split ; try split ; auto.
  Qed.

  Lemma impl_exists_longer : forall {a: asn} {va: val->asn},
      impl copred_defs ((asn_exists_ va) &*& a) (asn_exists v, ((va v) &*& a)).
  Proof.
    intros a va h H.
    apply impl_sepcon_comm in H.
    constructor.
    inversion H.
    destruct H2 as [h1 [h2 [H4 [H5 H6]]]].
    inversion H6.
    destruct H8 as [v H8].
    exists v.
    constructor.
    exists h2.
    exists h1.
    split; try split ; try assumption.
    rewrite heap_plus_commutative.
    assumption.
  Qed.

  Lemma impl_replace_part: forall {a1 a2 a2' a3 : asn}, 
      impl copred_defs a2 a2' ->
      impl copred_defs (a1 &*& a2 &*& a3) (a1 &*& a2' &*& a3).
  Proof.
    intros a1 a2 a2' a3 Ha2a2'.
    intros h H.
    constructor.
    inversion H.
    destruct H2 as [h1 [h23 [H_eq [Hh1 Hh23]]]].
    exists h1.
    exists h23.
    split ; try split ; try assumption.
    constructor.
    inversion Hh23.
    subst.
    destruct H5 as [h2 [h3 [Heq' [Hh2 Hh3]]]].
    exists h2.
    exists h3.
    split ; try split ; try assumption.
    apply Ha2a2'.
    assumption.
  Qed.

  Lemma impl_replace_back:
    forall {a1 a2 a2': asn},
      impl copred_defs a2 a2' ->
      impl copred_defs (a1 &*& a2) (a1 &*& a2').
  Proof.
    intros a1 a2 a2'.
    apply impl_sepcon.
  Qed.

  Lemma impl_sepcon_comm_2nd : forall {a1 a2 a3: asn},
      impl copred_defs (a1 &*& a2 &*& a3) (a1 &*& a3 &*& a2).
  Proof.
    intros.
    apply (impl_replace_back impl_sepcon_comm).
  Qed.


  Lemma impl_replace_front:
    forall {a1 a1' a2: asn},
      impl copred_defs a1 a1' ->
      impl copred_defs (a1 &*& a2) (a1' &*& a2).
  Proof.
    intros a1 a1' a2.
    intros H.
    intros h H2.
    apply impl_sepcon_comm.
    apply (@impl_replace_back a2 a1 a1').
    - assumption.
    - apply impl_sepcon_comm.
      assumption.
  Qed.

  Lemma impl_remove_emp_front :
    forall {a: asn},
      impl copred_defs (emp &*& a) a.
  Proof.
    intros a h H.
    inversion H.
    destruct H2 as [h1 [h2 [H_eq [Hh1 Hh2]]]].
    inversion Hh1.
    subst.
    rewrite heap_plus_empty in H_eq.
    rewrite heap_eq_iff_eq in H_eq.
    rewrite H_eq.
    assumption.
  Qed.

  Lemma impl_add_emp_front :
    forall {a: asn},
      impl copred_defs a (emp &*& a).
  Proof.
    unfold impl.
    intros.
    constructor.
    exists emptyheap.
    exists h.
    split; try split.
    - apply heap_eq_iff_eq.
      rewrite heap_plus_empty.
      reflexivity.
    - constructor.
    - assumption.
  Qed.


  Fixpoint add_emp_back (a: asn) : asn :=
    match a with
    | a1 &*& a2 => a1 &*& (add_emp_back a2)
    | a => a &*& emp
    end
  .

  Lemma impl_add_emp_back:
    forall {a: asn},
      impl copred_defs (add_emp_back a) a
  .
  Proof.
    induction a ; try (
                      intros h mod_h;
                      inversion mod_h ; subst;
                      try (destruct H1 as [h1 [h2 [h_eq [mod_h1 mod_emp]]]]);
                      try (destruct H2 as [h1 [h2 [h_eq [mod_h1 mod_emp]]]]);
                      inversion mod_emp ; subst;
                      rewrite heap_plus_empty_right in h_eq;
                      apply heap_eq_iff_eq in h_eq;
                      rewrite h_eq;
                      assumption;
                      fail
                    ).
    intros h mod_h.
    constructor.
    simpl in mod_h.
    inversion mod_h ; subst.
    destruct H1 as [h1 [h2 [h_eq [mod_h1 mod_h2_emp]]]].
    exists h1.
    exists h2.
    split; try split ; try assumption.
    apply IHa2.
    assumption.
  Qed.

  Fixpoint remove_emp_back (a: asn) : asn :=
    match a with
    | a1 &*& emp => a1
    | a1 &*& a2 => a1 &*& (remove_emp_back a2)
    | a => a
    end
  .

  Lemma impl_remove_emp_back :
    forall {a: asn},
      impl copred_defs a (add_emp_back a)
  .
  Proof.
    induction a ; try (intros h H' ; apply impl_sepcon_comm ; apply impl_add_emp_front ; assumption).
    intros h H.
    inversion H.
    inversion H2 as [h1 [h2 [h_eq [mod_h1 mod_h2]]]].
    constructor.
    exists h1.
    exists h2.
    split ; try split ; try assumption.
    apply IHa2.
    assumption.
  Qed.

  Lemma impl_emp_eq : 
    forall (v1 v2: val),
    impl copred_defs (asn_val (binop_eq v1 v2)) emp.
  Proof.
    intros v1 v2 h H.
    inversion H.
    apply models_val.
  Qed.

  Lemma impl_add_eq :
    forall (v1 v2: val) {a: asn},
    impl copred_defs (asn_val (binop_eq v1 v2) &*& a) a.
  Proof.
    intros v1 v2 a h H.
    apply impl_remove_emp_front.
      eapply impl_replace_front.
      - intros h' H'.
        eapply impl_emp_eq ; try trivial.
        apply H'.
      - apply H.
  Qed.

  (* Note: proofs related to the combination of Asn.v and Hoare.v,
   * are in Hoare.v (otherwise we would need circular dependencies in
   * Coq).
   *)
    
End impl_proofs.
