(**
  Soundness proof of the input/output verification approach.
  Note: The proof rules do not support non-termination (but the step rules do).
 *)

Require Import ZArith.
Require Import Coq.Init.Datatypes.
(* If you want to use list notations:
 * Import Coq.Lists.List.ListNotations.
 * Local Open Scope list_scope.
 *)
Require Import Val.
Require Import Op.
Require Import Cmd.
Require Import Asn.
Require Import Hoare.
Require Import Trace.
Require Import Colist.
Require Import Heap.

Section soundness.
  Variable copred_defs: copred -> copred_def.
  Variable fc: (func -> (list val -> cmd)).

  Lemma soundness_exp: forall (f: val->val) (v:val),
      f v = val_bool true ->
      @valid copred_defs fc ({{ emp }}
                     cmd_val v
                     {{fun x : val => asn_val (f x)}}).
  Proof.
    intros f v1 H h modelsP tau v step.
    exists (inl h).
    inversion modelsP.
    split.
    * inversion step.
      apply trace_nil.
    * intro fin.
      inversion step.
      subst.
      rewrite H.
      apply models_val.
  Qed.

  Axiom excluded_middle : forall P: Prop, P + ~P.

  Ltac ExMid k :=
    let u := fresh "fin_" k in
    let v := fresh "notfin_" k in
    destruct (excluded_middle (finite_colist k)) as [u|v]
  .

  Lemma soundness_let: forall (P: asn) (Q1: val->asn) (Q: val->asn)
                              (c: cmd) (f: val->cmd),
      @proof copred_defs fc ({{P}}c{{Q1}}) ->
      (forall v:val, @proof copred_defs fc ({{Q1 v}}f v {{Q}})) ->
      (forall v : val, @valid copred_defs fc ({{Q1 v}}f v {{Q}})) ->
      @valid copred_defs fc ({{P}}c {{Q1}}) ->
      @valid copred_defs fc ({{P}}cmd_let c f {{Q}}).
  Proof.
    intros P Q1 Q c f prf p HvalidQ IHprf.
    intros h modelsP tau v step.
    inversion step.
    unfold valid_helper.
    destruct (IHprf h modelsP tau1 v1 H1) as [[h1 | []] [Hg1 Hg2]].
    * ExMid tau1.
    - destruct (HvalidQ v1 h1 (Hg2 fin_tau1) tau2 v H4) as [[h2 | []] H7].
      + (* all executions finite and no trace-contradicts. *)
        exists (inl h2).
        split.
        { apply trace_composition with (h0 := h1) ; intuition. }
        { intro smth.
          apply H7.
          apply append_fin' with (cl1:=tau1) (cl2:=tau2).
          auto.
        }
      + (* 2nd trace contradicts. *)
        exists (inr bottom_ctr).
        split; auto.
        apply traces_contradict_app with (h1 := h1) ; intuition.
    - (* 1st execution diverges. *)
      exists (inl h1).
      split.
      + rewrite (app_notfin_eq) ; assumption.
      + intros H_contra.
        contradict notfin_tau1.
        apply append_fin' with (cl1:=tau1) (cl2:=tau2).
        assumption.
        * (* 1st trace contradicts. *)
          exists (inr bottom_ctr).
          split;auto.
          inversion Hg1.
          rewrite colist_append_assoc_eq.
          rewrite colist_help_app_eq.
          apply (trace_contradict r r' h h2 tau0 tau3 (tau4+++tau2) b arg) ; assumption.
  Qed.

  Lemma soundness_app:
    forall (P: asn) (Q: val->asn) (f: func) (vs: list val)
           (prf: @proof copred_defs fc ({{P}}fc f vs {{Q}}))
           (IHprf : @valid copred_defs fc ({{P}}fc f vs {{Q}})),
      @valid copred_defs fc ({{P}}cmd_app f vs {{Q}})
  .
  Proof.
    intros P Q f vs prf IHprf.
    intros h modelsP tau v step.
    inversion step.
    apply (IHprf h modelsP tau0 v) in H3.
    inversion H3 as [g H4].
    exists g.
    destruct g.
    - split.
      + apply traces_add_no_io.
        apply H4.
      + destruct H4 as [H4 H5].
        intro fin_tau0'.
        apply H5.
        inversion fin_tau0'.
        assumption.
    - rewrite <- (@append_conil_eq action tau0).
      rewrite <- (colist_help_app_eq).
      split;auto.
      destruct b.
      apply traces_contradict_app with (h1 := h).
      rewrite <- (heap_plus_empty h).
      apply trace_frame.
      apply trace_no_io.
      apply H4.
  Qed.


  Lemma soundness_then:
    forall (P: asn) (Q: val->asn) (ct ce: cmd)
           (prf: @proof copred_defs fc ({{P}}ct {{Q}}))
           (IHprf: @valid copred_defs fc ({{P}}ct {{Q}})),
                       @valid copred_defs fc ({{P}}cmd_if (val_bool true) ct ce {{Q}}).
  Proof.
    intros P Q ct ce prf IHprf.
    intros h modelsP tau v step.
    inversion step.
    - apply IHprf ; assumption.
    - intuition.
  Qed.

  Lemma soundness_else:
    forall (P: asn) (Q: val->asn) (guard: val) (ct ce: cmd)
           (prf: @proof copred_defs fc ({{P}}ce {{Q}}))
           (IHprf: @valid copred_defs fc ({{P}}ce {{Q}}))
           (guard_not_true: guard <> val_bool true),
                       @valid copred_defs fc ({{P}}cmd_if guard ct ce {{Q}}).
  Proof.
    intros P Q guard ct ce prf IHprf guard_not_true.
    intros h modelsP tau v step.
    inversion step.
    - contradiction guard_not_true.
      symmetry.
      assumption.
    - apply IHprf ; assumption.
  Qed.

  Lemma soundness_frame: 
    forall (P R: asn) (Q: val->asn) (c: cmd)
           (prf : @proof copred_defs fc ({{P}}c {{Q}}))
           (IHprf : @valid copred_defs fc ({{P}}c {{Q}})),
      @valid copred_defs fc ({{P &*& R}}c {{fun v : val => Q v &*& R}}).
  Proof.
    intros P R Q c prf IHprf.
    intros h modelsPR tau v step.
    inversion modelsPR.
    destruct H1 as [h1 [h2 [h_eq [modelsP modelsR]]]].
    destruct (IHprf h1 modelsP tau v step) as [g IHprf'].
    apply heap_eq_iff_eq in h_eq.
    destruct g as [g | g_bot].
    - exists (inl (heap_plus g h2)).
      rewrite h_eq.
      split.
      + apply trace_frame.
        apply IHprf'.
      + intro tau_fin.
        constructor.
        exists g.
        exists h2.
        split ; try split ; try apply IHprf' ; assumption.
    - exists (inr bottom_ctr).
      destruct IHprf' as [IHprf' _].
      inversion IHprf' ; subst.
      split ; auto.
      apply trace_contradict with (r:=r) (h2:=(heap_plus h4 h2)) (tau2:=tau2).
      + assumption.
      + apply trace_frame.
        assumption.
  Qed.

  Lemma soundness_leak:
    forall (P R: asn) (Q: val->asn) (c: cmd)
           (prf: @proof copred_defs fc ({{P}}c {{fun v : val => Q v &*& R}}))
           (IHprf: @valid copred_defs fc ({{P}}c {{fun v : val => Q v &*& R}})),
      @valid copred_defs fc ({{P}} c {{Q}})
  .
  Proof.
    intros P R Q c prf IHprf.
    intros h modelsP tau v step.
    unfold valid in IHprf.
    unfold valid_helper.
    unfold valid_helper in IHprf.
    destruct (IHprf h modelsP tau v step) as [[g | g_bot] [tr IHprf']].    
    - ExMid tau.
      + apply IHprf' in fin_tau.
        inversion fin_tau.
        destruct H1 as [h_Q [h_R [g_eq [mod_Q mod_R]]]].
        exists (inl h_Q).
        split.
        * apply trace_leak with (hr := h_R).
          apply heap_eq_iff_eq in g_eq.
          rewrite <- g_eq.
          assumption.
        * intro fin_tau'.
          assumption.
      + exists (inl g).
        split ; try assumption.
        contradiction.
   - exists (inr g_bot).
     auto.
  Qed.

  Lemma soundness_bio:
    forall (b: Bio.bio) (t1 t2: val) (arg retval: val),
      @valid copred_defs fc
     ({{asn_token t1 &*& asn_bio b t1 arg retval t2}}cmd_bio b arg
      {{fun r : val => asn_val (binop_eq r retval) &*& asn_token t2}}).
  Proof.
    intros b t1 t2 arg retval.
    intros h modelsP tau v step.
    
    
    (* h equals {| token(t1), bio(...) |}. *)
    inversion modelsP.
    destruct H1 as [h1 [h2 [h_eq [models_t1 models_bio]]]].
    inversion models_t1.
    inversion models_bio.
    subst.
    rewrite heap_plus_one in h_eq.
    rewrite heap_eq_iff_eq in h_eq.
    
    (* This might be a trace where the world contradicts the prediction.
     * Of course we can't decide that constructively. *)
    destruct (excluded_middle (retval = v)) as [nicetrace | eviltrace].
    - exists (inl {| chunk_token t2 |} ).
      split.
      + rewrite h_eq.
        (* tau equals bio(...):::nil *)
        inversion step.
        subst.
        apply trace_bio.
      + intro tau_fin.
        apply models_sepcon.
        exists emptyheap.
        exists {| chunk_token t2 |}.
        rewrite heap_plus_empty.
        split; try split.
        * rewrite nicetrace.
          unfold binop_eq.
          destruct (val_eq_dec v v) ; try intuition.
          apply models_val.
        * apply models_token.
    - exists (inr bottom_ctr).
      split; auto.
      rewrite h_eq.
      inversion step.
      subst.
      rewrite <- append_conil_eq.
      eapply trace_contradict.
      + apply eviltrace.
      + rewrite append_conil_eq.
        apply trace_bio.
  Qed.

  Lemma soundness_no_op:
    forall (P: asn) (t1 t2: val) (c: cmd) (Q: val->asn)
           (prf: @proof copred_defs fc ({{asn_token t2 &*& P}}c {{Q}}))
           (IHprf : @valid copred_defs fc ({{asn_token t2 &*& P}}c {{Q}})),
      @valid copred_defs fc ({{asn_token t1 &*& asn_no_op t1 t2 &*& P}}c {{Q}})
  .
  Proof.
    intros P t1 t2 c Q prf IHprf.
    intros h modelsP tau v step.
    unfold valid_helper.

    inversion modelsP ; subst.
    destruct H1 as [ht1 [h_nop_P [h_eq [models_t1 models_nop_P]]]].
    inversion models_nop_P; subst.
    destruct H1 as [h_nop [h_P [h_nop_P_eq [models_nop models_P]]]].
    
    inversion models_t1; subst.
    inversion models_nop; subst.

    rewrite heap_eq_iff_eq in h_eq.
    rewrite h_eq.
    rewrite heap_eq_iff_eq in h_nop_P_eq.
    rewrite h_nop_P_eq.

    unfold valid in IHprf.
    unfold valid_helper in IHprf.
    assert (models copred_defs (heap_plus {|chunk_token t2|} h_P)  (asn_token t2 &*& P)) as models_t2_P.
    { apply models_sepcon.
      exists {|chunk_token t2|}.
      exists h_P.
      split; try split.
      - constructor.
      - assumption.
    }
    destruct (IHprf (heap_plus {|chunk_token t2|} h_P) models_t2_P tau v step)
      as [g [H_trace H_models]].
    destruct g as [g | g_evil].
    - exists (inl g).
      split.
      + rewrite <- (@append_conil_eq action tau).
        eapply trace_composition.
          rewrite <- heap_plus_assoc.
          eapply trace_frame.
          rewrite heap_plus_one.
          apply trace_no_op.
        * assumption.
      + assumption.
    - exists (inr bottom_ctr).
      split; auto.
      inversion H_trace.
      eapply trace_contradict.
      apply H.
      rewrite <- append_conil_eq.
      eapply trace_composition.
      + rewrite <- heap_plus_assoc.
        rewrite heap_plus_one.
        apply trace_frame.
        apply trace_no_op.
      + apply H0.
  Qed.

  Lemma soundness_rewrite_post_impl:
    forall (h g: heap) (a a': asn) (v: val) (tau: trace),
      traces h tau (inl g) ->
      impl copred_defs a a' ->
      (finite_colist tau -> models copred_defs g a) ->
      @valid_helper copred_defs h a' tau
  .
  Proof.
    intros h g Q Q' v tau Htr impl_Qv H'.
    exists (inl g).
    split; try assumption.
    intro fin_tau.
    apply impl_Qv.
    apply H'.
    assumption.  
  Qed.

  Lemma models_t2_t3:
    forall {t2 t3: val},
      models copred_defs {|chunk_token t2, chunk_token t3|} (asn_token t2 &*& asn_token t3)
  .
  Proof.
    constructor.
    exists {| chunk_token t2 |}.
    exists {| chunk_token t3 |}.
    split; try split ; try constructor.
    rewrite heap_plus_one.
    apply heap_eq_iff_eq.
    reflexivity. 
  Qed.

  Lemma soundness_rewrite_post_split:
    forall (h g : heap) (c : cmd) (P P' : asn) (Q Q' : val -> asn)
           (v : val) (tau : trace)
           (Htr : traces h tau (inl g))
           (t1 t2 t3: val)
           (eq_Qv : asn_token t1 &*& asn_split t1 t2 t3 &*& P = Q v)
           (eq_Q'v : asn_token t2 &*& asn_token t3 &*& P = Q' v)
           (H' : finite_colist tau -> models copred_defs g (Q v))
           (prf : @proof copred_defs fc ({{P'}}c {{Q}}))
           (IHprf : @valid copred_defs fc ({{P'}}c {{Q}})),
      @valid_helper copred_defs h (Q' v) tau
  .
  Proof.
    intros.
    ExMid tau.
   - assert (models copred_defs g (Q v)) as models_Qv. {
       apply H'. assumption.
     }
     rewrite <- eq_Qv in models_Qv.
     inversion models_Qv ; subst.
     destruct H1 as [h_t1 [h_split_P [g_eq [mod_t1 mod_split_P]]]].
     inversion mod_split_P; subst.
     destruct H1 as [h_split [h_P [h_split_P_eq [mod_split mod_P]]]].
     exists (inl (heap_plus {| chunk_token t2, chunk_token t3 |} h_P)).
     split.
     + rewrite heap_eq_iff_eq in g_eq.
       rewrite g_eq in Htr.
       rewrite <- (append_conil_back_eq action tau).
       eapply trace_composition.
       * apply Htr.
       * inversion mod_t1; subst.
         rewrite heap_eq_iff_eq in h_split_P_eq.
         rewrite h_split_P_eq.
         rewrite <- heap_plus_assoc.
         rewrite heap_plus_one.
         apply trace_frame.
         inversion mod_split.
         apply trace_split.
     + intro fin_tau'.
       rewrite <- eq_Q'v.
       constructor.
       exists {|chunk_token t2|}.
       exists (heap_plus {|chunk_token t3|} h_P).
       split; try split.
       * rewrite <- heap_plus_assoc.
         rewrite heap_plus_one.
         apply heap_eq_iff_eq.
         reflexivity.
       * constructor.
       * constructor.
         exists {|chunk_token t3|}.
         exists h_P.
         split; try split ; try constructor; try assumption.
   - unfold valid_helper.
     exists (inl g).
     split.
     + assumption.
     + contradiction.
  Qed.

  Lemma soundness_rewrite_post_join:
    forall (h g : heap) (c : cmd) (P P' : asn) (Q Q' : val -> asn)
           (v : val) (tau : trace)
           (Htr : traces h tau (inl g))
           (t1 t2 t3: val)
           (eq_Qv : asn_token t1 &*& asn_token t2 &*& asn_join t1 t2 t3 &*& P = Q v)
           (eq_Q'v : asn_token t3 &*& P = Q' v)
           (H' : finite_colist tau -> models copred_defs g (Q v))
           (prf : @proof copred_defs fc ({{P'}}c {{Q}}))
           (IHprf : @valid copred_defs fc ({{P'}}c {{Q}})),
      @valid_helper copred_defs h (Q' v) tau
  .
  Proof.
    intros.
    ExMid tau.
   - assert (models copred_defs g (Q v)) as models_Qv. {
       apply H'. assumption.
     }
     rewrite <- eq_Qv in models_Qv.
     inversion models_Qv ; subst.
     destruct H1 as [h_t1 [h_t2_join_P [g_eq [mod_t1 mod_t2_join_P]]]].
     inversion mod_t2_join_P; subst.
     destruct H1 as [h_t2 [h_join_P [h_t2_join_P_eq [mod_t2 mod_join_P]]]].
     inversion mod_join_P; subst.
     destruct H1 as [h_join [h_P [h_join_P_eq [mod_join mod_P]]]].
     exists (inl (heap_plus {| chunk_token t3 |} h_P)).
     split.
     + rewrite heap_eq_iff_eq in g_eq.
       rewrite g_eq in Htr.
       rewrite <- (append_conil_back_eq action tau).
       eapply trace_composition.
       * apply Htr.
       * inversion mod_t1; subst.
         inversion mod_t2; subst.
         rewrite heap_eq_iff_eq in h_t2_join_P_eq.
         rewrite h_t2_join_P_eq.
         rewrite heap_eq_iff_eq in h_join_P_eq.
         rewrite h_join_P_eq.
         inversion mod_join; subst.
         assert (heap_plus {|chunk_token t2|} (heap_plus {|chunk_join t1 t2 t3|} h_P)
                  = heap_plus ({|chunk_token t2, chunk_join t1 t2 t3|}) h_P) as H_eq.
         {
           rewrite <- heap_plus_assoc.
           rewrite heap_plus_one.
           reflexivity.
         }
         rewrite H_eq.
         rewrite <- heap_plus_assoc.
         rewrite heap_plus_one.
         apply trace_frame.
         apply trace_join.
     + intro fin_tau'.
       rewrite <- eq_Q'v.
       constructor.
       exists {|chunk_token t3|}.
       exists h_P.
       split; try split.
       * constructor. 
       * assumption.
   - unfold valid_helper.
     exists (inl g).
     split.
     + assumption.
     + contradiction.
  Qed.

  Lemma soundness_rewrite_post:
    forall (h g: heap) (c: cmd) (P': asn) (Q Q': val->asn) (v: val) (tau: trace)
      (Htr: traces h tau (inl g))
      (rewrite_Qv: @asn_rewrite copred_defs (Q v) (Q' v))
      (H': finite_colist tau -> models copred_defs g (Q v))
      (prf: @proof copred_defs fc ({{P'}}c {{Q}}))
      (IHprf: @valid copred_defs fc ({{P'}}c {{Q}})),
      @valid_helper copred_defs h (Q' v) tau
  .
  Proof.
    intros.
    remember (Q v) in rewrite_Qv.
    remember (Q' v) in rewrite_Qv.
    induction rewrite_Qv eqn:H ; subst.
    - eapply soundness_rewrite_post_impl; eauto.
    - eapply soundness_rewrite_post_split; eauto.
    - eapply soundness_rewrite_post_join; eauto.
  Qed.

  Lemma soundness_rewrite_impl:
   forall (P P': asn) (Q Q': val->asn) (c: cmd),
      @impl copred_defs P P' ->
      (forall v : val, @asn_rewrite copred_defs (Q v) (Q' v)) ->
      @proof copred_defs fc ({{P'}}c{{Q}}) ->
      @valid copred_defs fc ({{P'}}c{{Q}}) ->
      @valid copred_defs fc ({{P}}c{{Q'}})
  .
  Proof.
    intros P P' Q Q' c impl_PP' rewrite_Q prf IHprf.
    intros h modelsP tau v step.
    destruct (IHprf h (impl_PP' h modelsP) tau v step) as [g [Htr H']].
    destruct g as [g | []].
    - eapply soundness_rewrite_post; eauto.
    - exists (inr bottom_ctr).
      auto.
  Qed.

  Lemma soundness_rewrite_split:
    forall (P: asn) (Q Q': val->asn) (c: cmd) (t1 t2 t3: val),
      (forall v : val, @asn_rewrite copred_defs (Q v) (Q' v)) ->
      @proof copred_defs fc ({{asn_token t2 &*& asn_token t3 &*& P}}c{{Q}}) ->
      @valid copred_defs fc ({{asn_token t2 &*& asn_token t3 &*& P}}c{{Q}}) ->
      @valid copred_defs fc ({{asn_token t1 &*& asn_split t1 t2 t3 &*& P}}c{{Q'}})
  .
  Proof.
    intros P Q Q' c t1 t2 t3 rewrite_Q prf IHprf.
    intros h modelsP tau v step.
    inversion modelsP ; subst.
    destruct H1 as [h_t1 [h_split_P [h_eq [models_t1 models_split_P]]]].
    inversion models_t1.
    inversion models_split_P ; subst.
    destruct H3 as [h_split [h_P [h_split_P_eq [models_split models_P]]]].
    assert (models copred_defs (heap_plus {|chunk_token t2|} (heap_plus {|chunk_token t3|} h_P)) (asn_token t2 &*& asn_token t3 &*& P)) as H_mod.
    {
      constructor.
      exists {|chunk_token t2|}.
      exists (heap_plus {|chunk_token t3|} h_P).
      split; try split.
      - constructor.
      - constructor.
        exists {|chunk_token t3|}.
        exists h_P.
        split; try split.
        + constructor.
        + assumption.
    }
    destruct (IHprf (heap_plus {|chunk_token t2|} (heap_plus {|chunk_token t3|} h_P)) H_mod tau v step) as [g [Htr H']].
    apply heap_eq_iff_eq in h_eq.
    apply heap_eq_iff_eq in h_split_P_eq.
    inversion models_split.
    subst.
    rewrite <- heap_plus_assoc.
    rewrite heap_plus_one.
    rewrite <- (@append_conil_eq action tau).
    destruct g as [g | []].
    - apply soundness_rewrite_post with (g:=g) (Q:=Q) (c:=c) (P':=asn_token t2 &*& asn_token t3 &*& P); auto.
      + eapply trace_composition.
        * apply trace_frame.
          apply trace_split.
        * rewrite <- heap_plus_assoc in Htr.
          rewrite heap_plus_one in Htr.
          assumption.
      + rewrite (@append_conil_eq action tau).
        assumption.
    - exists (inr bottom_ctr).
      split; auto.
      rewrite (@append_conil_eq action tau).
      inversion Htr.
      eapply trace_contradict.
      * apply H.
      * rewrite <- heap_plus_assoc in H0.
        rewrite heap_plus_one in H0.
        rewrite <- append_conil_eq.
        eapply trace_composition.
        { apply trace_frame.
          apply trace_split.
        } 
        apply H0.
        
  Qed.

  Lemma soundness_rewrite_join:
    forall (P: asn) (Q Q': val->asn) (c: cmd) (t1 t2 t3: val),
      (forall v : val, @asn_rewrite copred_defs (Q v) (Q' v)) ->
      @proof copred_defs fc ({{asn_token t3 &*& P}}c{{Q}}) ->
      @valid copred_defs fc ({{asn_token t3 &*& P}}c{{Q}}) ->
      @valid copred_defs fc ({{asn_token t1 &*& asn_token t2 &*& asn_join t1 t2 t3 &*& P}}c{{Q'}})
  .
  Proof.
    intros P Q Q' c t1 t2 t3 rewrite_Q prf IHprf.
    intros h modelsP tau v step.
    inversion modelsP ; subst.
    destruct H1 as [h_t1 [h_t2_join_P [h_eq [models_t1 models_t2_join_P]]]].
    inversion models_t1.
    inversion models_t2_join_P ; subst.
    destruct H3 as [h_t2 [h_join_P [h_t2_join_P_eq [models_t2 models_join_P]]]].
    inversion models_join_P ; subst.
    destruct H1 as [h_join [h_P [h_join_P_eq [models_join models_P]]]].
    inversion models_join.
    assert (models copred_defs (heap_plus {|chunk_token t3|} h_P) (asn_token t3 &*& P)) as H_mod.
    {
      constructor.
      exists {|chunk_token t3|}.
      exists h_P.
      split; try split.
      - constructor.
      - assumption.
    }
    destruct (IHprf (heap_plus {|chunk_token t3|} h_P) H_mod tau v step) as [g [Htr H']].
    apply heap_eq_iff_eq in h_eq.
    apply heap_eq_iff_eq in h_t2_join_P_eq.
    apply heap_eq_iff_eq in h_join_P_eq.
    inversion models_t2.
    inversion models_join.
    subst.
    rewrite <- (heap_plus_assoc {|chunk_token t2|}).
    rewrite <- heap_plus_assoc.
    rewrite heap_plus_one with (c:=chunk_token t2).
    rewrite heap_plus_one.
    rewrite <- (@append_conil_eq action tau).
    destruct g as [g | []].
    - apply soundness_rewrite_post with (g:=g) (Q:=Q) (c:=c) (P':=asn_token t3 &*& P); auto.
      + eapply trace_composition.
        * apply trace_frame.
          apply trace_join.
        * assumption.
      + rewrite (@append_conil_eq action tau).
        assumption.
    - exists (inr bottom_ctr).
      split; auto.
      rewrite (@append_conil_eq action tau).
      inversion Htr.
      eapply trace_contradict.
      * apply H.
      * rewrite <- append_conil_eq.
        eapply trace_composition.
        { apply trace_frame.
          apply trace_join.
        } 
        apply H0.
        
  Qed.

  Lemma soundness_rewrite:
    forall (P P': asn) (Q Q': val->asn) (c: cmd),
      @asn_rewrite copred_defs P P' ->
      (forall v : val, @asn_rewrite copred_defs (Q v) (Q' v)) ->
      @proof copred_defs fc ({{P'}}c{{Q}}) ->
      @valid copred_defs fc ({{P'}}c{{Q}}) ->
      @valid copred_defs fc ({{P}}c{{Q'}})
  .
  Proof.
    intros P P' Q Q' c rewrite_P rewrite_Q prf IHprf.
    inversion rewrite_P.
    - eapply soundness_rewrite_impl; eauto.
    - subst; eapply soundness_rewrite_split; eauto.
    - subst; eapply soundness_rewrite_join; eauto.
  Qed.

  Lemma soundness_exists:
    forall (Q1 Q2: val->asn) (c: cmd),
      (forall v : val, @proof copred_defs fc ({{Q1 v}}c {{Q2}})) ->
      (forall v : val, @valid copred_defs fc ({{Q1 v}}c {{Q2}})) ->
      @valid copred_defs fc ({{asn_exists x, Q1 x}}c {{Q2}})
  .
  Proof.
    intros Q1 Q2 c H_proof H_valid.
    intros h modelsP tau v step.
    unfold valid in H_valid.
    inversion modelsP.
    destruct H1.
    apply H_valid with (v:=x) ; assumption.
  Qed.

  Lemma soundness_disj:
    forall (P1 P2: asn) (Q: val->asn) (c: cmd),
      @proof copred_defs fc ({{P1}}c {{Q}}) ->
      @proof copred_defs fc ({{P2}}c {{Q}}) ->
      @valid copred_defs fc ({{P1}}c {{Q}}) ->
      @valid copred_defs fc ({{P2}}c {{Q}}) ->
      @valid copred_defs fc ({{asn_disj P1 P2}}c {{Q}})
  .
  Proof.
    intros P1 P2 Q c prf1 prf2 IHprf1 IHprf2.
    intros h modelsP tau v step.
    inversion modelsP.
    - apply IHprf1 ; assumption.
    - apply IHprf2 ; assumption.
  Qed.

  Lemma soundness_skip:
    @valid copred_defs fc ({{emp}} cmd_skip {{fun _ => emp}}).
  Proof.
    intros h modelsP tau v step.
    inversion modelsP.
    exists (inl emptyheap).
    inversion step.
    split.
    - apply trace_nil.
    - intro H_fin.
      apply models_val.
  Qed.

  Lemma soundness_false:
    forall (c: cmd) (Q: val->asn),
      @valid copred_defs fc ({{asn_val (val_bool false)}} c {{Q}}).
  Proof.
    intros c Q h modelsP tau v step.
    inversion modelsP.
  Qed.

  (** Main soundness proof. *)
  Theorem soundness: forall (PcQ: hoare_triple),
      @proof copred_defs fc (PcQ) -> @valid copred_defs fc PcQ.
  Proof.
    intros PcQ prf.
    induction prf.
    - apply soundness_exp; auto.
    - eapply soundness_let; eauto.
    - apply soundness_app; auto.
    - apply soundness_then; auto.
    - apply soundness_else; auto.
    - apply soundness_frame; auto.
    - eapply soundness_leak; eauto.
    - apply soundness_bio; auto.
    - apply soundness_no_op; auto.
    - eapply soundness_rewrite; eauto.
    - apply soundness_exists; auto.
    - apply soundness_disj; auto.
    - apply soundness_skip.
    - apply soundness_false.
  Qed.


End soundness.
