Require Import Colist.
Require Import Val.
Require Import Bio.
Require Import Heap.

(** An instance of an event that is observable by the environment,
    e.g. "write the number 7" or "turn on the light". *)
Inductive action :=
| action_ctr: bio -> val -> val -> action
| action_no_io
.

(** An execution of a program yields a trace of I/O events that have
    happen. The trace can be infinite or finite. *)
Definition trace := colist action.

Definition emptytrace: trace := conil.

Inductive bottom := bottom_ctr.

(** A relation expressing that a given heap allows producing
    a trace, resulting in another heap or bottom (in case
    the environment contradicts the predictions in the initial heap).
*)
Inductive traces : heap -> trace -> (heap + bottom) -> Prop :=
| trace_bio: forall (t1 t2: val) (b: bio) (arg: val) (ret: val),
    traces
      {| chunk_token t1, chunk_bio b t1 arg ret t2 |}
      ((action_ctr b arg ret):::conil)
      (inl {| chunk_token t2 |})
| trace_nil: traces emptyheap conil (inl emptyheap)
| trace_frame: forall (h h' hr: heap) (tau: trace),
    traces h tau (inl h') ->
    traces (heap_plus h hr) tau (inl (heap_plus h' hr))
| trace_composition: forall (h h0 h': heap) (tau1 tau2: trace),
    traces h tau1 (inl h0) -> traces h0 tau2 (inl h') ->
    traces h (tau1 +++ tau2) (inl h')
| trace_leak: forall (h h' hr: heap) (tau: trace),
    traces h tau (inl (heap_plus h' hr)) ->
    traces h tau (inl h')
| trace_split: forall (t1 t2 t3: val),
    traces
      {| chunk_token t1, chunk_split t1 t2 t3 |}
      conil
      (inl {| chunk_token t2, chunk_token t3 |})
| trace_join: forall (t1 t2 t3: val),
    traces
      {| chunk_token t1, chunk_token t2, chunk_join t1 t2 t3 |}
      conil
      (inl {| chunk_token t3 |})
| trace_contradict: forall (r r': val) (h1 h2: heap)
                           (tau1 tau2 tau3: trace) (b: bio) (arg: val),
    r <> r' ->
    traces h1 (tau1 +++ (action_ctr b arg r):::tau2) (inl h2) ->
    traces h1 (tau1 +++ (action_ctr b arg r'):::tau3) (inr bottom_ctr)
| trace_no_io: traces emptyheap (action_no_io:::conil) (inl emptyheap)
| trace_no_op: forall (t1 t2: val),
    traces {| chunk_token t1, chunk_no_op t1 t2 |}
           conil
           (inl {| chunk_token t2 |})
.

Lemma traces_contradict_app: forall (tau1 tau2: trace) (h h1: heap),
    traces h tau1 (inl h1) ->
    traces h1 tau2 (inr bottom_ctr) ->
    traces h (tau1 +++ tau2) (inr bottom_ctr)
.
Proof.
  intros tau1 tau2 h h1 H H7.
  inversion H7.
  rewrite <- colist_append_assoc_eq.
  apply trace_contradict with (r:=r) (h2:=h2) (tau2:=tau3).
  - assumption.
  - rewrite colist_append_assoc_eq.
    apply trace_composition with (h0 := h1).
    + assumption.
    + assumption.
Qed.

Lemma traces_add_no_io: forall (tau: trace) (h h1: heap),
    traces h tau (inl h1) ->
    traces h (action_no_io:::tau) (inl h1)
.
Proof.
  intros tau h h1 tr.
  rewrite <- (@append_conil_eq action tau).
  rewrite <- (colist_help_app_eq).
  apply trace_composition with (h0 := h).
  - rewrite <- (heap_plus_empty h).
    apply trace_frame.
    apply trace_no_io.
  - assumption.
Qed.
